function spider_init()

	storage.generator = game.create_random_generator()
	storage.entropy = {
		value = 0,
		maxEvents = 120,
		events = 0
	}

	storage.aggroDistanceSq = 2100
	storage.interpolationDistance = 8
	storage.damageCap = 185
	storage.fakeEntities = {}

	local spiderProperties = {
		{ -- spider1
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({{2885.8,-5.4},{2800.3,-1.8},{2301.8,0.7},{854.3,143.2},{26.8,138.2},{-115.7,38.2},{-843.2,-134.3},{-990.7,-369.3},{-1208.2,-321.8},{-1721.1,-259.2}})
			},
			attackArea = {
				left_top =     {x = -2613, y = -400},
				right_bottom = {x = -1721, y = 400}
			},
			patrolArea = {
				left_top     = {x = -2141, y = -398},
				right_bottom = {x = -1600, y = 398}
			},
			minPoi = {
				patrol = 8,
				attack = 10
			},
			patrolsToTriggerAttack = 14,
			mitigation = -1.8,
			lowHealth = 0.35,
			lowAmmo = 650,
			fullAmmo = 800,
			ammoType = "explosive-rocket"
		},
		{ -- spider2
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({{2951.3,4.9},{2271.0,2.4},{978.3,158.6},{325.7,62.9},{-72.5,-221.8},{-644.5,-95.9},{-871.3,-239.5},{-1057.2,-356.4}})
			},
			attackArea = {
				left_top =     {x = -1970, y = -400},
				right_bottom = {x = -1050, y = 400}
			},
			patrolArea = {
				left_top     = {x = -1070, y = -398},
				right_bottom = {x = -1040, y = -250}
			},
			minPoi = {
				patrol = 3,
				attack = 12
			},
			patrolsToTriggerAttack = 745,
			mitigation = 0.32,
			lowHealth = 0.7,
			lowAmmo = 400,
			fullAmmo = 800,
			ammoType = "explosive-rocket"
		},
		{ -- spider3
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({{2960.2,1.7},{2297.2,-1.4},{1435.7,102.5},{886.1,-125.8},{504.9,-114.8},{293.9,-10.9},{158.3,-52.7}})
			},
			attackArea = {
				left_top =     {x = -1030,  y = -400},
				right_bottom = {x = 160,   y = 400}
			},
			patrolArea = {
				left_top     = {x = -510, y = -398},
				right_bottom = {x = 172,   y = 300}
			},
			minPoi = {
				patrol = 7,
				attack = 12
			},
			patrolsToTriggerAttack = 40,
			mitigation = 0.65,
			lowHealth = 0.5,
			lowAmmo = 350,
			fullAmmo = 800,
			ammoType = "explosive-rocket"
		},
		{ -- spider4
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({{2961.6,4.7},{2307.6,2.7},{1189.6,154.7},{289.6,92.7},{35.6,128.7},{-108.1,43.0}})
			},
			attackArea = {
				left_top =     {x = -1030,  y = -330},
				right_bottom = {x = 162,   y = 400}
			},
			patrolArea = {
				left_top     = {x = -510, y = -398},
				right_bottom = {x = 172,   y = 398}
			},
			minPoi = {
				patrol = 7,
				attack = 12
			},
			patrolsToTriggerAttack = 35,
			mitigation = 0.65,
			lowHealth = 0.5,
			lowAmmo = 250,
			fullAmmo = 800,
			ammoType = "explosive-rocket"
		},
		{ -- spider5
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({{2966.0,-0.8},{2302.3,2.2},{1954.1,-192.3},{1489.4,-190.6}})
			},
			attackArea = {
				left_top =     {x = -35,  y = -400},
				right_bottom = {x = 1500,   y = 400}
			},
			patrolArea = {
				left_top     = {x = 887, y = -398},
				right_bottom = {x = 1502,   y = 398}
			},
			minPoi = {
				patrol = 8,
				attack = 13
			},
			patrolsToTriggerAttack = 25,
			mitigation = 0.83,
			lowHealth = 0.7,
			lowAmmo = 200,
			fullAmmo = 600,
			ammoType = "explosive-rocket"
		},
		{ -- spider6
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({{2959.7,2.2},{2420.0,-0.8},{2078.7,-22}})
			},
			attackArea = {
				left_top =     {x = 700,  y = -400},
				right_bottom = {x = 2098,   y = 400}
			},
			patrolArea = {
				left_top     = {x = 1600, y = -398},
				right_bottom = {x = 2100,   y = 398}
			},
			minPoi = {
				patrol = 8,
				attack = 14
			},
			patrolsToTriggerAttack = 22,
			mitigation = 0.88,
			lowHealth = 0.65,
			lowAmmo = 200,
			fullAmmo = 800,
			ammoType = "explosive-rocket"
		},
		{ -- spider7
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({{2962.9,-1.8},{2916.6,-2.5}})
			},
			attackArea = {
				left_top =     {x = 2000,  y = -400},
				right_bottom = {x = 3000,   y = 400}
			},
			patrolArea = {
				left_top     = {x = 2300, y = -398},
				right_bottom = {x = 3000,   y = 398}
			},
			minPoi = {
				patrol = 8,
				attack = 13
			},
			patrolsToTriggerAttack = 10,
			mitigation = 0.89,
			lowHealth = 0.85,
			lowAmmo = 5500,
			fullAmmo = 6500,
			ammoType = "rocket"
		},
		{ -- spider8
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({{2962.9,1.8},{2916.6,2.5}})
			},
			attackArea = {
				left_top =     {x = 1400,  y = -400},
				right_bottom = {x = 3000,   y = 400}
			},
			patrolArea = {
				left_top     = {x = 2300, y = -398},
				right_bottom = {x = 3000,   y = 398}
			},
			minPoi = {
				patrol = 8,
				attack = 12
			},
			patrolsToTriggerAttack = 15,
			mitigation = 0.89,
			lowHealth = 0.75,
			lowAmmo = 5500,
			fullAmmo = 6500,
			ammoType = "rocket"
		},
		{ -- spider9
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({ {2968.8,2.1},{2909.9,2.4} })
			},			
			attackArea = {
				left_top =     {x = 2150,  y = -400},
				right_bottom = {x = 3000,   y = 400}
			},
			patrolArea = {
				left_top     = {x = 2300, y = -398},
				right_bottom = {x = 3000,   y = 398}
			},
			minPoi = {
				patrol = 7,
				attack = 9
			},
			patrolsToTriggerAttack = 20,
			mitigation = 0.92,
			lowHealth = 0.95,
			lowAmmo = 1400,
			fullAmmo = 2400,
			ammoType = "rocket"
		},
		{ -- spider10
			route = {
				attack = {},
				patrol = {},
				commute = interpolate({ {2968.8,-2.1},{2909.9,-2.4} })
			},
			attackArea = {
				left_top =     {x = 2140,  y = -400},
				right_bottom = {x = 3000,   y = 400}
			},
			patrolArea = {
				left_top     = {x = 2250, y = -398},
				right_bottom = {x = 3000,   y = 398}
			},
			minPoi = {
				patrol = 7,
				attack = 9
			},
			patrolsToTriggerAttack = 20,
			mitigation = 0.92,
			lowHealth = 0.95,
			lowAmmo = 1400,
			fullAmmo = 2400,
			ammoType = "rocket"
		}
	}
	
	storage.enemySpiders = {}
	
	local p1, p2 = {}
	local spider
	for n = 1, #spiderProperties do
		spider = game.get_entity_by_tag("enemySpider" .. n)
		storage.enemySpiders[spider.unit_number] = spiderProperties[n]
		storage.enemySpiders[spider.unit_number].number = n
		storage.enemySpiders[spider.unit_number].unit_number = spider.unit_number
		storage.enemySpiders[spider.unit_number].stuckTime = 0
		storage.enemySpiders[spider.unit_number].goingHome = false
		storage.enemySpiders[spider.unit_number].spiderObject = spider
		storage.enemySpiders[spider.unit_number].aggro = false
		storage.enemySpiders[spider.unit_number].aggroTimer = 0
		storage.enemySpiders[spider.unit_number].canSeeMines = false
		storage.enemySpiders[spider.unit_number].patrolsDone = 0
		storage.enemySpiders[spider.unit_number].patrolWaypointID = 1
		storage.enemySpiders[spider.unit_number].attackWaypointID = 1
		storage.enemySpiders[spider.unit_number].commuteWaypointID = #storage.enemySpiders[spider.unit_number].route.commute
		storage.enemySpiders[spider.unit_number].damageTracking = {tick = -1, damageTypes = {}}
		spider.grid.inhibit_movement_bonus = true  -- Start with exos disabled
		
		--[[
		for i = 1, #storage.enemySpiders[spider.unit_number].route.patrol do
			p1 = storage.enemySpiders[spider.unit_number].route.patrol[i].position
			p2 = storage.enemySpiders[spider.unit_number].route.patrol[1].position

			if isLinearlyTraversable(p1, p2) then
				storage.enemySpiders[spider.unit_number].route.patrol[i].traversable = true
				-- rendering.draw_line{surface = game.surfaces[1], from = p1, to = p2, color = {0, 1, 0}, width = 6}
			else
				storage.enemySpiders[spider.unit_number].route.patrol[i].traversable = false
				-- rendering.draw_line{surface = game.surfaces[1], from = p1, to = p2, color = {0, 0, 1}, width = 6}
			end
			
		end
		
		for i = 1, #storage.enemySpiders[spider.unit_number].route.attack do
			p1 = storage.enemySpiders[spider.unit_number].route.attack[i].position
			p2 = storage.enemySpiders[spider.unit_number].route.attack[1].position
			
			if isLinearlyTraversable(p1, p2) then
				storage.enemySpiders[spider.unit_number].route.attack[i].traversable = true
				-- rendering.draw_line{surface = game.surfaces[1], from = p1, to = p2, color = {0, 0.7, 0}, width = 6}
			else
				storage.enemySpiders[spider.unit_number].route.attack[i].traversable = false
				-- rendering.draw_line{surface = game.surfaces[1], from = p1, to = p2, color = {0, 0, 0.7}, width = 6}
			end
			
		end	
		]]

	end
	
	loadPoi()
	-- generatePoi()
	
	for n = 1, #spiderProperties do
		spider = game.get_entity_by_tag("enemySpider" .. n)
		storage.enemySpiders[spider.unit_number].route.patrol = explodeRoute(generateRoute(n, "patrol"))
		storage.enemySpiders[spider.unit_number].route.attack = explodeRoute(generateRoute(n, "attack"))
	end
	
	
	-- Track path requests for stuck spiders
	storage.pathRequests = {}
	
	-- Track player spiders
	storage.playerSpiders = {}
	
	-- Create "fake-player" force to cause spiders to also attack non-military targets
	local fakePlayer = game.create_force("fake-player")
	game.forces[4].set_cease_fire(game.forces[1], true)
	game.forces[1].set_cease_fire(game.forces[4], true)
	
	
end


function explodeRoute(route)
	-- Expects a list of POIs.  Returns the interpolated route
	local explodedRoute = {}
	local previousPoi = route[1]
	for step, poi in pairs(route) do
		if step > 1 then
			for waypoint, waypointData in pairs(storage.poi[previousPoi].reachablePoi[poi]) do
				if waypoint > 1 then
					explodedRoute[#explodedRoute + 1] = {
						position = waypointData.position,
						traversable = waypointData.traversable[route[1]]
					}
				end
			end
			previousPoi = poi
		end
	end
	return explodedRoute
end


function generateRoute(spiderNumber, routeType)
	local spiderUnitNumber = game.get_entity_by_tag("enemySpider" .. spiderNumber).unit_number
	local spider = storage.enemySpiders[spiderUnitNumber]
	
	local area = {}
	local minPoi = 0
	if routeType == "attack" then
		area = spider.attackArea
		minPoi = spider.minPoi.attack
	else
		area = spider.patrolArea
		minPoi = spider.minPoi.patrol
	end
	
	local poiInArea = getPoiInArea(area)
	
	-- To do - compute breadcrumbs.toBattle
	
	local route = pickNextPoi(poiInArea, {spiderNumber}, minPoi)
	local routeLength = 0

	while #route < minPoi or route[#route] ~= route[1] do
		route = pickNextPoi(poiInArea, route, minPoi)
		if route[#route] ~= route[1] and #route > minPoi * 3 then
			-- We're lost!  Retrace our steps before the route gets too crazy
			routeLength = #route
			game.print("Retracing steps, route length exceeded 3x minimum: " .. routeLength)
			for step = routeLength - 1, 1, -1 do
				route[#route + 1] = route[step]
			end
		end
	end

	-- game.print("Route with " .. #route .. " steps: " .. serpent.line(route))
	-- drawRoute(route)
	return route
end


function getPoiInArea(area)
	local poiList = {}
	for id, poi in pairs(storage.poi) do
		if positionInArea(poi.position, area) then
			poiList[#poiList+1] = id
		end
	end
	return poiList
end


function pickNextPoi(availablePoi, routeSoFar, minPoi)
	-- Expects a table of POI within the spider's area, and the route so far.  Returns the route with a new POI added.
	
	local currentPoi  = routeSoFar[#routeSoFar]
	local previousPoi = routeSoFar[#routeSoFar - 1]		-- Will be nil if there is only one POI in the route so far
		
	local route = routeSoFar
	local connections = {}
	
	local baseScore = 100
	local totalScore = 0
	
	for poi, value in pairs(storage.poi[currentPoi].reachablePoi) do
		for key, inAreaPoi in pairs(availablePoi) do
			if poi == inAreaPoi then
				connections[poi] = baseScore
				totalScore = totalScore + baseScore
			end
		end
	end
	
	
	local shortestDistance = 10
	local shortRouteSteps = {}
	
	if #routeSoFar >= minPoi then
		-- If the commute endpoint is available as a connection, go back there and we're done making the route
		if connections[routeSoFar[1]] then
			route[#route + 1] = route[1]
			return route
		end
		
		-- Otherwise, find the shortest distance to get home
		for poi, score in pairs(connections) do
			if storage.poi[poi].breadcrumbs.toCommuteEndpoint[route[1]] and storage.poi[poi].breadcrumbs.toCommuteEndpoint[route[1]] < shortestDistance then
				shortestDistance = storage.poi[poi].breadcrumbs.toCommuteEndpoint[route[1]]
			end
		end
		-- Find all the connections with that shortest distance
		for poi, score in pairs(connections) do
			if storage.poi[poi].breadcrumbs.toCommuteEndpoint[route[1]] == shortestDistance then
				-- Prioritize those connections that will get us home faster
				connections[poi] = connections[poi] + 1000
				totalScore = totalScore + 1000
			end
		end
	end
	
	-- Reduce the score of the previously visited Point of Interest
	if previousPoi then
		connections[previousPoi] = connections[previousPoi] - 99
		totalScore = totalScore - 99
	end
	
	-- Pick a connection with a weighted random score
	local randomNumber = storage.generator(totalScore)
	local cumulativeScore = 0
	local chosenConnection
	
	for poi, score in pairs(connections) do
		cumulativeScore = cumulativeScore + score
		if cumulativeScore >= randomNumber then
			chosenConnection = poi
			break
		end
	end
	
	route[#route + 1] = chosenConnection
	
	--game.print("Here are the connections (and their scores) that I could reach from my current POI: ")
	--showTable(connections)
	--game.print("I choose connection " .. chosenConnection)
	--game.print("I will return this: " .. serpent.line(route))
	
	return route
end


function generatePoi()
	
	-- Create a list of Points of Interest that a spider might like to visit
	storage.poi = {}
	
	-- Add the spider commute endpoints as the first 10 entries
	local commute
	local spiderUnitNumber = {}
	for spiderNumber = 1, 10 do
		spiderUnitNumber[spiderNumber] = game.get_entity_by_tag("enemySpider" .. spiderNumber).unit_number
		commute = storage.enemySpiders[spiderUnitNumber[spiderNumber]].route.commute
		storage.poi[spiderNumber] = commute[#commute]
	end
	
	-- Add the rest of the POI
	addRawPoi()

	
	-- Compute traversability from this poi to other poi
	local spiderPatrolArea = {}
	local spiderAttackArea = {}
	local maxConnectionDistanceSq = 600^2
	
	for thisPoi = 1, #storage.poi do
		storage.poi[thisPoi].reachablePoi = {}
		p1 = storage.poi[thisPoi].position
		
		for otherPoi = 1, #storage.poi do
			if thisPoi ~= otherPoi then
				p2 = storage.poi[otherPoi].position
				if distanceSq(p1, p2) < maxConnectionDistanceSq then
					if isLinearlyTraversable(p1, p2) or isLinearlyTraversable(p2, p1) then
						storage.poi[thisPoi].reachablePoi[otherPoi] = interpolate({{p1.x, p1.y}, {p2.x, p2.y}})
						for n, waypoint in pairs(storage.poi[thisPoi].reachablePoi[otherPoi]) do
							storage.poi[thisPoi].reachablePoi[otherPoi][n].traversable = {}
							for spiderNumber = 1, 10 do
								spiderPatrolArea = storage.enemySpiders[spiderUnitNumber[spiderNumber]].patrolArea
								spiderAttackArea = storage.enemySpiders[spiderUnitNumber[spiderNumber]].attackArea
								if positionInArea(waypoint.position, spiderPatrolArea) or positionInArea(waypoint.position, spiderAttackArea) then 
									if isLinearlyTraversable(storage.poi[spiderNumber].position, waypoint.position) then
										storage.poi[thisPoi].reachablePoi[otherPoi][n].traversable[spiderNumber] = true
									end
								end
							end
						end
					end
				end
			end
		end
	end
	
	local poiInAreaWithoutBreadcrumb
	local distanceFromGoal
	local lowestDistance
	
	for poi, data in pairs(storage.poi) do
		storage.poi[poi].breadcrumbs = {
			toCommuteEndpoint = {},
			toBattle = 9
		}
	end
	
	-- Compute breadcrumbs.toCommuteEndpoint[spiderNumber]
	for spiderNumber = 1, 10 do
		distanceFromGoal = 0
		storage.poi[spiderNumber].breadcrumbs.toCommuteEndpoint[spiderNumber] = distanceFromGoal
		
		spiderAttackArea = storage.enemySpiders[spiderUnitNumber[spiderNumber]].attackArea
		poiInAreaWithoutBreadcrumb = getPoiInArea(spiderAttackArea)		
		
		while #poiInAreaWithoutBreadcrumb > 0 do
			for id, poi in pairs(poiInAreaWithoutBreadcrumb) do
				-- Check neighbours, find lowest distance
				lowestDistance = 50
				for neighbour, data in pairs(storage.poi[poi].reachablePoi) do
					if positionInArea(storage.poi[neighbour].position, spiderAttackArea) and storage.poi[neighbour].breadcrumbs.toCommuteEndpoint[spiderNumber] and storage.poi[neighbour].breadcrumbs.toCommuteEndpoint[spiderNumber] < lowestDistance then
						lowestDistance = storage.poi[neighbour].breadcrumbs.toCommuteEndpoint[spiderNumber]
					end
				end
				-- Add the new distance to this POI
				if lowestDistance < 50 then
					storage.poi[poi].breadcrumbs.toCommuteEndpoint[spiderNumber] = lowestDistance + 1
				end
				-- Remove the POI from the list of in-area POIs if it already has breadcrumbs to the commute endpoint
				if storage.poi[poi].breadcrumbs.toCommuteEndpoint[spiderNumber] then
					poiInAreaWithoutBreadcrumb[id] = nil
				end
			end
		end
	end
	
end


function addRawPoi()
	-- West of Lake Tall
	storage.poi[11] = {position = {x = -2531, y = -159}}
	storage.poi[12] = {position = {x = -2288, y = -126}}
	storage.poi[13] = {position = {x = -2350, y = 70}}
	storage.poi[14] = {position = {x = -2530, y = -340}}
	storage.poi[15] = {position = {x = -2612, y = -274}}
	storage.poi[16] = {position = {x = -2336, y = -358}}
	storage.poi[17] = {position = {x = -2183, y = -220}}
	storage.poi[18] = {position = {x = -2085, y = -361}}
	storage.poi[19] = {position = {x = -2563, y = 247}}
	storage.poi[20] = {position = {x = -2258, y = 293}}
	storage.poi[21] = {position = {x = -2347, y = 210}}
	storage.poi[22] = {position = {x = -2118, y = 72}}
	storage.poi[23] = {position = {x = -1884, y = 251}}
	storage.poi[24] = {position = {x = -1887, y = 387}}
	storage.poi[25] = {position = {x = -1724, y = 313}}
	storage.poi[26] = {position = {x = -1713, y = 70}}
	storage.poi[27] = {position = {x = -1909, y = 50}}
	storage.poi[28] = {position = {x = -1836, y = -42}}
	storage.poi[29] = {position = {x = -1743, y = -100}}
	storage.poi[30] = {position = {x = -2088, y = -73}}
	storage.poi[31] = {position = {x = -1682, y = 396}}
	storage.poi[32] = {position = {x = -1389, y = 204}}
	storage.poi[33] = {position = {x = -1232, y = 392}}
	storage.poi[34] = {position = {x = -1260, y = 293}}
	storage.poi[35] = {position = {x = -1123, y = 276}}
	storage.poi[36] = {position = {x = -1233, y = -93}}
	storage.poi[37] = {position = {x = -1408, y = -261}}
	storage.poi[38] = {position = {x = -1269, y = -356}}
	storage.poi[39] = {position = {x = -1165, y = 25}}
	storage.poi[40] = {position = {x = -1325, y = -55}}
	storage.poi[41] = {position = {x = -1453, y = 57}}
	storage.poi[42] = {position = {x = -1565, y = -96}}
	storage.poi[43] = {position = {x = -2140, y = 377}}
	storage.poi[44] = {position = {x = -1948, y = -83}}
	storage.poi[45] = {position = {x = -1059, y = -316}}
	storage.poi[46] = {position = {x = -1057, y = -397}}
	
	
	
	-- East of Lake Tall
	storage.poi[47] = {position = {x = -947, y = -206}}
	storage.poi[48] = {position = {x = -868, y = -269}}
	storage.poi[49] = {position = {x = -946, y = -37}}
	storage.poi[50] = {position = {x = -1003, y = 40}}
	storage.poi[51] = {position = {x = -952, y = 182}}
	storage.poi[52] = {position = {x = -985, y = 236}}
	storage.poi[53] = {position = {x = -961, y = 389}}
	storage.poi[54] = {position = {x = -724, y = 339}}
	storage.poi[55] = {position = {x = -508, y = 287}}
	storage.poi[56] = {position = {x = -664, y = 178}}
	storage.poi[57] = {position = {x = -813, y = 79}}
	storage.poi[58] = {position = {x = -657, y = 36}}
	storage.poi[59] = {position = {x = -813, y = -59}}
	storage.poi[60] = {position = {x = -742, y = -156}}
	storage.poi[61] = {position = {x = -603, y = -160}}
	storage.poi[62] = {position = {x = -645, y = -390}}
	storage.poi[63] = {position = {x = -470, y = -386}}
	storage.poi[64] = {position = {x = -460, y = -273}}
	storage.poi[65] = {position = {x = -239, y = -262}}
	storage.poi[66] = {position = {x = -174, y = -215}}
	storage.poi[67] = {position = {x = -164, y = 48}}
	storage.poi[68] = {position = {x = -279, y = 14}}
	storage.poi[69] = {position = {x = -437, y = 56}}
	storage.poi[70] = {position = {x = -446, y = 159}}
	storage.poi[71] = {position = {x = -255, y = 205}}
	storage.poi[72] = {position = {x = -59, y = 204}}
	storage.poi[73] = {position = {x = -10, y = 77}}
	storage.poi[74] = {position = {x = 32, y = -65}}
	storage.poi[75] = {position = {x = 170, y = -345}}
	storage.poi[76] = {position = {x = 18, y = -311}}
	storage.poi[77] = {position = {x = -325, y = 397}}
	storage.poi[78] = {position = {x = -51, y = 328}}
	
	-- East of Spider Lakes
	storage.poi[79] = {position = {x = 286, y = 254}}
	storage.poi[80] = {position = {x = 258, y = -111}}
	storage.poi[81] = {position = {x = 241, y = -255}}
	storage.poi[82] = {position = {x = 351, y = 397}}
	storage.poi[83] = {position = {x = 586, y = 28}}
	storage.poi[84] = {position = {x = 701, y = 102}}
	storage.poi[85] = {position = {x = 557, y = -126}}
	storage.poi[86] = {position = {x = 646, y = -359}}
	storage.poi[87] = {position = {x = 425, y = -394}}
	storage.poi[88] = {position = {x = 554, y = 346}}
	storage.poi[89] = {position = {x = 714, y = -208}}
	
	-- Desert
	storage.poi[90] = {position = {x = 846, y = -342}}
	storage.poi[91] = {position = {x = 841, y = -211}}
	storage.poi[92] = {position = {x = 853, y = -56}}
	storage.poi[93] = {position = {x = 856, y = 87}}
	storage.poi[94] = {position = {x = 897, y = 357}}
	storage.poi[95] = {position = {x = 991, y = 224}}
	storage.poi[96] = {position = {x = 1094, y = 141}}
	storage.poi[97] = {position = {x = 1138, y = 42}}
	storage.poi[98] = {position = {x = 1126, y = -330}}
	storage.poi[99] = {position = {x = 1249, y = -366}}
	storage.poi[100] = {position = {x = 1375, y = -394}}
	storage.poi[101] = {position = {x = 1333, y = -245}}
	storage.poi[102] = {position = {x = 1228, y = -28}}
	storage.poi[103] = {position = {x = 1420, y = -82}}
	storage.poi[104] = {position = {x = 1241, y = 360}}
	storage.poi[105] = {position = {x = 1451, y = 342}}
	storage.poi[106] = {position = {x = 1615, y = 151}}
	storage.poi[107] = {position = {x = 1692, y = 392}}
	storage.poi[108] = {position = {x = 1728, y = 293}}
	storage.poi[109] = {position = {x = 1826, y = 188}}
	storage.poi[110] = {position = {x = 1912, y = -49}}
	storage.poi[111] = {position = {x = 1647, y = -237}}
	storage.poi[112] = {position = {x = 1816, y = -315}}
	storage.poi[113] = {position = {x = 1940, y = -390}}
	
	-- Final Base
	storage.poi[114] = {position = {x = 2091, y = -386}}
	storage.poi[115] = {position = {x = 2098, y = -250}}
	storage.poi[116] = {position = {x = 2214, y = -142}}
	storage.poi[117] = {position = {x = 2182, y = -44}}
	storage.poi[118] = {position = {x = 2080, y = 64}}
	storage.poi[119] = {position = {x = 2070, y = 155}}
	storage.poi[120] = {position = {x = 2185, y = 291}}
	storage.poi[121] = {position = {x = 2080, y = 392}}
	storage.poi[122] = {position = {x = 2617, y = 62}}
	storage.poi[123] = {position = {x = 2423, y = 314}}
	storage.poi[124] = {position = {x = 2348, y = 227}}
	storage.poi[125] = {position = {x = 2398, y = 182}}
	storage.poi[126] = {position = {x = 2448, y = 225}}
	storage.poi[127] = {position = {x = 2398, y = 278}}
	storage.poi[128] = {position = {x = 2947, y = 394}}
	storage.poi[129] = {position = {x = 2986, y = 351}}
	storage.poi[130] = {position = {x = 2995, y = 219}}
	storage.poi[131] = {position = {x = 2845, y = 113}}
	storage.poi[132] = {position = {x = 2729, y = -54}}
	storage.poi[133] = {position = {x = 2859, y = -1}}
	storage.poi[134] = {position = {x = 2847, y = -150}}
	storage.poi[135] = {position = {x = 2990, y = -184}}
	storage.poi[136] = {position = {x = 2993, y = -347}}
	storage.poi[137] = {position = {x = 2945, y = -390}}
	storage.poi[138] = {position = {x = 2757, y = -292}}
	storage.poi[139] = {position = {x = 2398, y = -396}}
	storage.poi[140] = {position = {x = 2400, y = -317}}
	storage.poi[141] = {position = {x = 2343, y = -274}}
	storage.poi[142] = {position = {x = 2397, y = -222}}
	storage.poi[143] = {position = {x = 2448, y = -272}}
	storage.poi[144] = {position = {x = 2921, y = -83}}
	storage.poi[145] = {position = {x = 2972, y = -117}}
	storage.poi[146] = {position = {x = 2969, y = 121}}
	storage.poi[147] = {position = {x = 2916, y = 85}}
	storage.poi[148] = {position = {x = 2519, y = 395}}

	-- Filling in some dead zones
	storage.poi[149] = {position = {x = 584, y = 217}}
	storage.poi[150] = {position = {x = -798, y = -384}}
	storage.poi[151] = {position = {x = -1074, y = -205}}
	storage.poi[152] = {position = {x = -1033, y = -85}}
	storage.poi[153] = {position = {x = -1068, y = 227}}
	storage.poi[154] = {position = {x = -1663, y = -382}}
	storage.poi[155] = {position = {x = -1740, y = -350}}
	storage.poi[156] = {position = {x = -1825, y = -396}}
	storage.poi[157] = {position = {x = 2716, y = 225}}
end


function distanceSq(p1, p2)
	return (p2.x - p1.x)^2 + (p2.y - p1.y)^2
end


function positionInArea(pos, area)
	return pos.x >= area.left_top.x and pos.x <= area.right_bottom.x and pos.y >= area.left_top.y and pos.y <= area.right_bottom.y
end


function nonTraversable(p)
	if p.x ~= p.x or p.y ~= p.y then
		log("Fishy goings on")
		return true
	end
	
    local waterTiles = game.surfaces[1].count_tiles_filtered{
        area = {
			left_top     = {p.x - 0.5, p.y - 0.5},
			right_bottom = {p.x + 1.5, p.y + 1.5}
		},
        collision_mask = {water_tile = true}
    }
    return waterTiles == 9
end


function isLinearlyTraversable(p1, p2)
    if nonTraversable(p1) then return false end
    if nonTraversable(p2) then return false end
	if samePosition(p1, p2) then return true end

    local dx = p2.x - p1.x
    local dy = p2.y - p1.y

    local midblocks, deltaX, deltaY, mbx, mby = 0
    local blockNum = 0
    local step = 3
	
    if math.abs(dx) > math.abs(dy) then
        midBlocks = math.ceil(math.abs(dx) / step)
        deltaY = dy / midBlocks
        if p2.x < p1.x then step = step * -1 end
        for mbx = p1.x, p2.x, step do
            blockNum = blockNum + 1
            mby = p1.y + (blockNum * deltaY)
            if nonTraversable({x = mbx, y = mby}) then return false end
        end
    else
        midBlocks = math.ceil(math.abs(dy) / step)
        deltaX = dx / midBlocks
        if p2.y < p1.y then step = step * -1 end
        for mby = p1.y, p2.y, step do
            blockNum = blockNum + 1
            mbx = p1.x + (blockNum * deltaX)
            if nonTraversable({x = mbx, y = mby}) then return false end
        end
    end
	
	
    return true
end


function spider_on_tick()
	-- Called every tick.
	for id, spider in pairs(storage.enemySpiders) do
		if spider.spiderObject and spider.spiderObject.valid == true then
			if spider.spiderObject.active then -- Perform some checks on each spider
				for_each_spider_on_tick(spider)
			end
		else
			-- Spider is dead, remove from list
			storage.enemySpiders[id] = nil
		end
	end
end


function for_each_spider_on_tick(spider)
	if spider.commuteWaypointID == 0 and spider.spiderObject.autopilot_destination == nil then
		-- I'm waiting back at base
		-- How much health, ammo and battery do I have?
		local ammoInLaunchers = spider.spiderObject.get_inventory(defines.inventory.spider_ammo).get_item_count(spider.ammoType)
		local ammoInTrunk     = spider.spiderObject.get_inventory(defines.inventory.spider_trunk).get_item_count(spider.ammoType)
		local spiderAmmo      = ammoInLaunchers + ammoInTrunk
		
		local batteryLevel = 1
		if spider.spiderObject.grid.battery_capacity > 0 then
			batteryLevel  = spider.spiderObject.grid.available_in_batteries / spider.spiderObject.grid.battery_capacity
		end
		
		if spider.spiderObject.get_health_ratio() == 1 and spiderAmmo >= spider.fullAmmo and batteryLevel > 0.9 then
			-- I have full health, ammo and battery again, lets commute!
			spider.spiderObject.add_autopilot_destination(spider.route.commute[1].position)
			spider.commuteWaypointID = 1
		end
		
		-- I'm not full ammo or health.  Should I just go anyway?
		if repairPossible(spider.route.commute[1].position) then
			-- Things are available, I should wait
		else
			-- There's something wrong with the network.  I should just go
			spider.spiderObject.add_autopilot_destination(spider.route.commute[1].position)
			spider.commuteWaypointID = 1
		end
	else
		-- I'm not waiting back at base.  Am I stuck?
		unstick(spider)
	end
	
	if spider.aggro == true then
		-- I have been aggro'd.  I'd better keep track of how long I've been like this otherwise I might get stuck and stay angry forever...
		spider.aggroTimer = spider.aggroTimer + 1
		
		-- Should we abort the aggro move?
		local ammoInLaunchers = spider.spiderObject.get_inventory(defines.inventory.spider_ammo).get_item_count(spider.ammoType)
		local deaggro = false
		if ammoInLaunchers > 0 then			-- Assuming we're not out of ammo and repair is actually possible
			local playerLasers = game.surfaces[1].find_entities_filtered{
				position = spider.spiderObject.position,
				radius = 24,
				force = "player",
				name = "laser-turret"
			}
			for id, laser in pairs(playerLasers) do
				if laser.is_connected_to_electric_network() then
					-- Lasers are live, deaggro
					deaggro = true
				end
			end
			
			local playerGuns = game.surfaces[1].find_entities_filtered{
				position = spider.spiderObject.position,
				radius = 18,
				force = "player",
				name = "gun-turret"
			}
			for id, gun in pairs(playerGuns) do
				if gun.get_inventory(defines.inventory.turret_ammo).get_item_count() > 0 then
					-- Guns are live, do not aggro
					deaggro = true
				end
			end
			
			local playerLandMines = 0
			if spider.canSeeMines then
				playerLandMines = game.surfaces[1].count_entities_filtered{
					position = spider.spiderObject.position,
					radius = 15,
					force = "player",
					name = "land-mine",
					limit = 1
				}
			end
			
			if playerLandMines > 0 then
				deaggro = true
			end
			
			
			
			if spider.spiderObject.get_health_ratio() < spider.lowHealth and repairPossible(spider.route.commute[1].position) then
				deaggro = true	-- Or if we're low health and we could go back for repairs
				spider.spiderObject.grid.inhibit_movement_bonus = false	-- Turn on exoskeletons
			end
		end
		
		if spider.aggroTimer > 360 or deaggro == true then
			-- I've been trying to get to that stupid player for 6 seconds now, time to give up.  Or maybe there's a turret nearby.
			local temp = spider.spiderObject.autopilot_destinations
			spider.spiderObject.autopilot_destination = nil
			spider.spiderObject.add_autopilot_destination(temp[2])
			spider.aggro = false
			spider.aggroTimer = 0
			spider.spiderObject.grid.inhibit_movement_bonus = true	-- Turn off exoskeletons
		end
		
	end
	
	-- Find player stuff
	if spider.spiderObject.position.x < storage.mostEastPlayerObject + 150 then	-- Only if we're west enough to be near player objects
		if nearbyBlocksHaveStuff(spider.spiderObject.position) then
			shootNonMilitary(spider)
		end
	end

	

	
end


local on_chunk_charted = function(event)
	if storage.entropy == nil then return end -- Required to avoid game crash in editor
	if storage.entropy.events < 0 then return end
	storage.entropy.events = storage.entropy.events + 1
	storage.entropy.value = math.abs(((event.position.x * storage.entropy.value) - event.position.y - event.tick)) % 10000000000
end


function djb2(int)
	-- Very simple hashing function
	local str = tostring(int)
	local hash = 5381
	for i = 1, #str do
		hash = ((hash * 33) + string.byte(str, i)) % 4294967296
	end
	return hash
end 


function repairPossible(repairSpot)

	local network = game.forces["enemy"].find_logistic_network_by_position(repairSpot, game.surfaces[1])
	
	if  network
	and network.valid
	and network.get_item_count("repair-pack") > 50
	and network.all_construction_robots > 10
	and network.find_cell_closest_to(repairSpot).owner.energy > 500
	then
		return true
	else
		return false
	end

end


function getBlockID(pos)	-- Returns the ID of the block the given position is in
	return math.floor(15 * math.floor((pos[1] + 3000) / 50) / 120) + ((15 * math.floor((pos[1] + 3000) / 50)) % 120) + (math.floor((pos[2] + 400) / 50) * 120)
end


function nearbyBlocksHaveStuff(pos)	-- Returns true if any of the 12 blocks around and to the west of the position contain player stuff
	local blockX = 0
	local blockY = 0
	
	for offsetX = -100, 50, 50 do
		blockX = pos.x + offsetX
		if blockX >= -3000 and blockX < 3000 then
			for offsetY = -50, 50, 50 do
				blockY = pos.y + offsetY
				if blockY >= -400 and blockY < 400 then
					if storage.blocks[getBlockID({blockX, blockY})].hasPlayerStuff then
						-- game.print(getBlockID({blockX, blockY}) .. " has player stuff, returning true")
						return true
					end
				end
			end
		end
	end
	return false	
end


function shootNonMilitary(spider)
	-- Look for nearby player military stuff, but exclude land mines
	local militaryStuff = game.surfaces[1].count_entities_filtered{
		position = spider.spiderObject.position,
		radius = 22,
		limit = 1,
		
		invert = true,
		name = "land-mine",
		force = {"enemy", "neutral"},
		is_military_target = false
	}
	
	if militaryStuff == 0 then
		
		local playerMines = {}
		if spider.canSeeMines then
			playerMines = game.surfaces[1].find_entities_filtered{
				position = spider.spiderObject.position,
				radius = 22,
				name = "land-mine",
				force = "player",
				limit = 1
			}
		end
			
		local playerStuff = game.surfaces[1].find_entities_filtered{
			position = spider.spiderObject.position,
			radius = 22,
			force = "player",
			collision_mask = {object = true},
			is_military_target = false,
			limit = 1
		}

		
		if #playerStuff > 0 or #playerMines > 0 then
			local playerObject = playerStuff[1] or playerMines[1]
			local objectPosition = playerObject.position
			
			-- Verify there isn't already a fake-entity here
			local existingFakeEntity = game.surfaces[1].find_entities_filtered{
				name = "simple-entity-with-force",
				area = {
					left_top = {
						x = objectPosition.x - 0.5,
						y = objectPosition.y - 0.5
					},
					right_bottom = {
						x = objectPosition.x + 0.5,
						y = objectPosition.y + 0.5
					}
				},
				limit = 1
			}
			if existingFakeEntity[1] then return end
			
			-- First teleport the player object out the way
			playerObject.teleport{
				x = playerObject.position.x - 1,
				y = playerObject.position.y
			}
			-- Then create an invisible fake entity at the object's original position
			local fake_entity = game.surfaces[1].create_entity{
				name = "simple-entity-with-force",
				force = "fake-player",
				position = objectPosition,
				create_build_effect_smoke = false
			}
			fake_entity.render_to_forces = "enemy"
			-- Add this fake entity to the list of fakeEntities
			storage.fakeEntities[#storage.fakeEntities + 1] = fake_entity
			
			-- Then teleport the player object back, so that it is "on top" for tooltip purposes
			playerObject.teleport{objectPosition.x, objectPosition.y}
		end
	end
end

function removeFakeEntities()
	-- Remove any fake entities that are not close to a spider
	for id, fakeEntity in pairs(storage.fakeEntities) do
		if fakeEntity.valid then
			--game.print(game.tick .. ":  Fake Entity ID: " .. id .. ": " .. serpent.line(fakeEntity.position))
			local closeSpiders = 0
			for unit_number, spiderProperties in pairs(storage.enemySpiders) do
				closeSpiders = closeSpiders + 1
				if distanceSq(spiderProperties.spiderObject.position, fakeEntity.position) > 1225 then
					closeSpiders = closeSpiders - 1
				end
			end
			if closeSpiders == 0 then
				fakeEntity.destroy()
				table.remove(storage.fakeEntities, id)
			end
		else
			table.remove(storage.fakeEntities, id)
		end
	end
end


local on_spider_command_completed = function(event)
	local spider = event.vehicle

	if (spider == nil) then return end -- not the droid you are looking for
	if (spider.valid == false) then return end -- destroyed/busy
	if (storage.enemySpiders[spider.unit_number] == nil) then return end	-- No data for this spider.  Probably a player spider.

	local spiderProperties = storage.enemySpiders[spider.unit_number]

	-- Was I aggrod on the previous waypoint?
	if spiderProperties.aggro == true then
		-- I was aggroed.  I just ran toward the player, I already have a waypoint queued to run back
		spiderProperties.aggro = false
		spiderProperties.aggroTimer = 0
		spider.grid.inhibit_movement_bonus = true	-- Turn off exoskeletons
		return
	end
	
	-- Am I getting unstuck right now?
	if spiderProperties.pathRequestID then
		-- I am getting unstuck right now.
		if #spider.autopilot_destinations > 0 then
			-- I'm not done with getting unstuck yet
			return
		else
			-- I'm done getting unstuck now.
			spiderProperties.pathRequestID = nil
		end
	end

	-- How much ammo and health do I have?
	local ammoInLaunchers = spider.get_inventory(defines.inventory.spider_ammo).get_item_count(spiderProperties.ammoType)
	local ammoInTrunk     = spider.get_inventory(defines.inventory.spider_trunk).get_item_count(spiderProperties.ammoType)
	local spiderAmmo      = ammoInLaunchers + ammoInTrunk
	local spiderHealth    = spider.get_health_ratio()
	
	local batteryLevel = 1
	if spider.grid.battery_capacity > 0 then
		batteryLevel  = spider.grid.available_in_batteries / spider.grid.battery_capacity
	end
	


	-- Am I home?
	if spiderProperties.commuteWaypointID == 0 then
		-- I just arrived home
		spider.grid.inhibit_movement_bonus = true	-- Turn off exoskeletons
		spiderProperties.canSeeMines = false	-- Disable land mine detection
		--game.print("canSeeMines set to false")
		
		-- Generate new patrol and attack routes
		spiderProperties.route.patrol = explodeRoute(generateRoute(spiderProperties.number, "patrol"))
		spiderProperties.route.attack = explodeRoute(generateRoute(spiderProperties.number, "attack"))
		spiderProperties.patrolWaypointID = 1
		spiderProperties.attackWaypointID = 1
		
		-- Is my health/ammo/battery not full?
		spiderProperties.goingHome = false
		if spiderHealth < 1 or spiderAmmo < spiderProperties.fullAmmo or batteryLevel < 0.9 then
			return	-- Not full, I should stop and wait.
		end
	end
	
	if spiderProperties.goingHome == true then
		goingHome(spiderProperties)
		return
	end
	
	-- I'm out and about.  Is my health, ammo or battery low?
	if spiderHealth < spiderProperties.lowHealth or spiderAmmo <= spiderProperties.lowAmmo or batteryLevel < 0.15 then
		-- Is repair/resupply even possible ?
		
		local repairSpot = spiderProperties.route.commute[1].position
		local network    = game.forces["enemy"].find_logistic_network_by_position(repairSpot, game.surfaces[1])
		
		if  network 
		and network.valid
		and network.get_item_count("repair-pack") > 50
		and network.get_item_count(spiderProperties.ammoType) > (spiderProperties.fullAmmo - spiderProperties.lowAmmo)
		and network.all_logistic_robots > 10
		and network.all_construction_robots > 10
		and network.find_cell_closest_to(repairSpot).owner.energy > 500
		then -- I'm going home
			goingHome(spiderProperties)
			spiderProperties.goingHome = true
			return
		end
	end
	
	-- Nothing wrong with me, lets head out!
	notGoingHome(spiderProperties)
	
end


function goingHome(spider)
	-- We are given a table of spider data for a spider that needs to find its way home
	
	
	-- Should we disable the exoskeleton?
	local nearbyPlayerObject = game.surfaces[1].count_entities_filtered{
		position = spider.spiderObject.position,
		radius = 40,
		force = "player",
		limit = 1
	}
	
	if nearbyPlayerObject == 1 then
		spider.spiderObject.grid.inhibit_movement_bonus = false		-- Run home fast
	else
		spider.spiderObject.grid.inhibit_movement_bonus = true		-- We've escaped, conserve energy
	end
	
	-- Attacking?
	if spider.patrolsDone == spider.patrolsToTriggerAttack then
		if spider.route.attack[spider.attackWaypointID].traversable then	-- Can we traverse directly back to waypoint 1?
			spider.attackWaypointID = 1
			spider.patrolsDone = 0
			spider.spiderObject.add_autopilot_destination(spider.route.attack[1].position)
			spider.spiderObject.grid.inhibit_movement_bonus = false		-- Always run during traversal
			return
		end
		
		-- Find the closest way back
		local halfwayOnAttack = #spider.route.attack / 2
		if spider.attackWaypointID <= halfwayOnAttack then
			-- The closest way back is behind me.  Go to previous waypoint on the attack route.
			if spider.attackWaypointID == 1 then
				spider.patrolsDone = 0
			else
				spider.attackWaypointID = spider.attackWaypointID - 1
				spider.spiderObject.add_autopilot_destination(spider.route.attack[spider.attackWaypointID].position)
				return
			end
		else
			-- The closest way back is in front of me.  Go to the next waypoint on the attack route.
			if spider.attackWaypointID == #spider.route.attack then
				-- If we reached the last waypoint in the attack route, the next waypoint is actually the first
				spider.attackWaypointID = 1
				spider.patrolsDone = 0
			else
				spider.attackWaypointID = spider.attackWaypointID + 1
				spider.spiderObject.add_autopilot_destination(spider.route.attack[spider.attackWaypointID].position)
				return
			end
		end
	end
	
	-- Commuting?
	if spider.patrolWaypointID == 1 then
		-- Yes, go to previous commute waypoint.
		spider.spiderObject.add_autopilot_destination(spider.route.commute[spider.commuteWaypointID].position)
		spider.commuteWaypointID = spider.commuteWaypointID - 1
		return
	end
	
	-- Patrolling?  
	if spider.route.patrol[spider.patrolWaypointID].traversable then	-- Can we traverse directly back to waypoint 1?
		spider.patrolWaypointID = 1
		spider.spiderObject.add_autopilot_destination(spider.route.patrol[1].position)
		spider.spiderObject.grid.inhibit_movement_bonus = false		-- Always run during traversal
		return
	end
	
	local halfwayOnPatrol = #spider.route.patrol / 2		-- To go home, I need to find the fastest way back to patrolWaypointID 1.
	if spider.patrolWaypointID <= halfwayOnPatrol then
		-- The closest way back is behind me.  Go to previous waypoint on the patrol.
		spider.patrolWaypointID = spider.patrolWaypointID - 1
	else
		-- The closest way back is in front of me.  Go to next waypoint on the patrol.
		spider.patrolWaypointID = spider.patrolWaypointID + 1
		-- Did we reach the end of the patrol route yet?
		if #spider.route.patrol < spider.patrolWaypointID then
			-- Yeah that's us reached the end, so the next waypoint is actually the first one in the list.
			spider.patrolWaypointID = 1
		end
	end
	-- Move toward the start of the patrol path
	spider.spiderObject.add_autopilot_destination(spider.route.patrol[spider.patrolWaypointID].position)
end


function aggro(entity, spider)
	-- Should we aggro on this entity?
	
	local playerLasers = game.surfaces[1].find_entities_filtered{
		position = entity.position,
		radius = 24,
		force = "player",
		name = "laser-turret"
	}
	for id, laser in pairs(playerLasers) do
		if laser.is_connected_to_electric_network() then
			-- Lasers are live, do not aggro
			return false
		end
	end
	
	local playerGuns = game.surfaces[1].find_entities_filtered{
		position = entity.position,
		radius = 18,
		force = "player",
		name = "gun-turret"
	}
	for id, gun in pairs(playerGuns) do
		if gun.get_inventory(defines.inventory.turret_ammo).get_item_count() > 0 then
			-- Guns are live, do not aggro
			return false
		end
	end
	
	
	-- A player is nearby!  Aggro on them.
	spider.aggro = true
	-- Turn Exoskeleton on
	spider.spiderObject.grid.inhibit_movement_bonus = false
	-- Move toward the player
	spider.spiderObject.add_autopilot_destination(entity.position)
	
	-- If we're commuting, retreat to the current commute waypoint
	if #spider.route.commute > spider.commuteWaypointID then
		spider.spiderObject.add_autopilot_destination(spider.route.commute[spider.commuteWaypointID].position)
		return true
	end
	
	-- If we're attacking, retreat to the current attack waypoint
	if spider.patrolsDone == spider.patrolsToTriggerAttack then
		spider.spiderObject.add_autopilot_destination(spider.route.attack[spider.attackWaypointID].position)
		return true
	end
	
	-- Otherwise we're patroling, retreat to the current patrol waypoint
	if spider == nil then game.print("Warning, spider was nil") end
	if spider.spiderObject == nil then game.print("Warning, spider.spiderObject was nil") end
	if spider.route == nil then game.print("Warning, spider.route was nil") end
	if spider.route.patrol == nil then game.print("Warning, spider.route.patrol was nil") end
	if spider.patrolWaypointID == nil then game.print("Warning, spider.patrolWaypointID was nil") end
	if spider.route.patrol[spider.patrolWaypointID] == nil then game.print("Warning, spider.route.patrol[spider.patrolWaypointID] was nil") end
	if spider.route.patrol[spider.patrolWaypointID].position == nil then game.print("Warning, spider.route.patrol[spider.patrolWaypointID].position was nil") end
	
	spider.spiderObject.add_autopilot_destination(spider.route.patrol[spider.patrolWaypointID].position)
	return true
end


function updateSeed()
	local hash = djb2(storage.entropy.value)
	storage.generator.re_seed(hash)
	storage.entropy.events = -1
	local spider
	
	for n = 2, 10 do
		spider = game.get_entity_by_tag("enemySpider" .. n)
		storage.enemySpiders[spider.unit_number].route.patrol = explodeRoute(generateRoute(n, "patrol"))
		storage.enemySpiders[spider.unit_number].route.attack = explodeRoute(generateRoute(n, "attack"))
	end
	spider = game.get_entity_by_tag("enemySpider" .. 1)
	storage.enemySpiders[spider.unit_number].patrolsDone = storage.enemySpiders[spider.unit_number].patrolsToTriggerAttack - 1
	
	-- game.print("Seed updated.  New seed: " .. hash)
end


function notGoingHome(spider)
	-- We are given a table of spider data for a spider that is fine and well, and needs to move out, patrol and/or attack.
	
	-- Is it time to update the seed yet?
	if storage.entropy.events >= storage.entropy.maxEvents then 
		updateSeed()
	end
	
	local distance
	local targetsInRange = {}
	-- Add players in aggro range
	for i, player in pairs(game.connected_players) do
		if player.character and player.character.valid then
			distance = (player.character.position.x - spider.spiderObject.position.x)^2 + (player.character.position.y - spider.spiderObject.position.y)^2
			if distance < storage.aggroDistanceSq then
				targetsInRange[math.floor(distance)] = player.character
			end
		end
	end

	-- Add playerSpiders in aggro range
	for i, playerSpider in pairs(storage.playerSpiders) do
		if playerSpider and playerSpider.valid then
			distance = (playerSpider.position.x - spider.spiderObject.position.x)^2 + (playerSpider.position.y - spider.spiderObject.position.y)^2
			if distance < storage.aggroDistanceSq then
				targetsInRange[math.floor(distance)] = playerSpider
			end
		else
			-- Spider is no longer valid, probably destroyed.  Remove from the list
			storage.playerSpiders[i] = nil
			-- game.print("Removed a spider from the list of player spiders")
		end
	end

	
	for dist, target in pairs(targetsInRange) do	-- Factorio Pairs returns the first 1024 values in guaranteed order, further away targets in an arbitrary order
		if aggro(target, spider) then return end		-- So targets that are especially close are prioritized, further away targets are chosen in arbitrary order
	end

	
	
	-- Commuting?
	if #spider.route.commute > spider.commuteWaypointID then
		-- Still on the way to the patrol area.  Move to the next commute waypoint.
		
		local forwardID  = math.max(1, (spider.commuteWaypointID + 1) % (#spider.route.commute + 1))
		local backwardID = math.max(1, (spider.commuteWaypointID - 1) % (#spider.route.commute + 1))
		local forward  = spider.route.commute[forwardID].position
		local backward = spider.route.commute[backwardID].position
		
		if spiderToAdvance(spider, forward, backward) then
			-- It's best to go forwards
			spider.commuteWaypointID = forwardID
		else
			-- It's best to go backwards
			spider.commuteWaypointID = backwardID
		end
		
		spider.spiderObject.add_autopilot_destination(spider.route.commute[spider.commuteWaypointID].position)
		return
	end
	
	-- Attacking?
	if spider.patrolsDone == spider.patrolsToTriggerAttack then
		if #spider.route.attack == spider.attackWaypointID then
			-- Finished our attack.  Generate new routes and go back to patrolling.
			spider.patrolsDone = 0
			spider.attackWaypointID = 1
			spider.patrolWaypointID = 1
			spider.route.patrol = explodeRoute(generateRoute(spider.number, "patrol"))
			spider.route.attack = explodeRoute(generateRoute(spider.number, "attack"))
			--game.print("Finished attack")
		else
			-- Not finished the attack yet, continue			
			local forwardID  = math.max(1, (spider.attackWaypointID + 1) % (#spider.route.attack + 1))
			local backwardID = math.max(1, (spider.attackWaypointID - 1) % (#spider.route.attack + 1))
			local forward  = spider.route.attack[forwardID].position
			local backward = spider.route.attack[backwardID].position
			
			if spiderToAdvance(spider, forward, backward) then
				-- It's best to go forwards
				spider.attackWaypointID = forwardID
			else
				-- It's best to go backwards
				spider.attackWaypointID = backwardID
			end
			
			--game.print("We attacking now!  [gps=" .. spider.spiderObject.position.x .. "," ..  spider.spiderObject.position.y .. "]")
			spider.spiderObject.add_autopilot_destination(spider.route.attack[spider.attackWaypointID].position)
			return
		end
	end
	
	-- Patrolling?
	local forwardID  = math.max(1, (spider.patrolWaypointID + 1) % (#spider.route.patrol + 1))
	local backwardID = math.max(1, (spider.patrolWaypointID - 1) % (#spider.route.patrol + 1))
	local forward  = spider.route.patrol[forwardID].position
	local backward = spider.route.patrol[backwardID].position
	
	if spiderToAdvance(spider, forward, backward) then
		-- It's best to go forwards
		spider.patrolWaypointID = forwardID
	else
		-- It's best to go backwards
		spider.patrolWaypointID = backwardID
	end

	
	
	if spider.patrolWaypointID == #spider.route.patrol then	-- Did I reach the end of the patrol route yet?
		-- Yes I've reached the end of the patrol, increment counter
		spider.patrolsDone = spider.patrolsDone + 1
	end
	
	-- Move to the next patrol waypoint.
	-- game.print(serpent.line(spider.route.patrol[spider.patrolWaypointID]))
	spider.spiderObject.add_autopilot_destination(spider.route.patrol[spider.patrolWaypointID].position)
end


function samePosition(p1, p2)
	return p1.x == p2.x and p1.y == p2.y
end


function spiderToAdvance(spider, forw, back)
	-- We are given a spider, its next waypoint, and its previous waypoint.
	-- We must return true if it's best/safest for the spider to go forwards, or false if it's best/safest for it to go backwards.
	
	if spider.spiderObject.position.x > storage.mostEastPlayerObject + 150 then
		-- The players haven't come this far yet, no need to run any scans
		return true
	end
	
	if not nearbyBlocksHaveStuff(spider.spiderObject.position) then return true end		-- The players don't have any objects in this area, dont bother looking for them
	
	-- local forw = {x = forward[1],  y = forward[2]}
	-- local back = {x = backward[1], y = backward[2]}
	
	-- Look for Turrets
	local playerTurrets = game.surfaces[1].find_entities_filtered{
		position = spider.spiderObject.position,
		radius = 32,
		force = "player",
		name = {"laser-turret", "gun-turret", "flamethrower-turret"},
		limit = 1
	}
	local playerLandMines = {}
	-- Look for landmines if we can see them
	if spider.canSeeMines then
		playerLandMines = game.surfaces[1].find_entities_filtered{
			position = spider.spiderObject.position,
			radius = 15,
			force = "player",
			name = {"land-mine"},
			limit = 1
		}
	end
	
	if #playerTurrets > 0 or #playerLandMines > 0 then
		local ent = playerTurrets[1] or playerLandMines[1]
		local runTo = furthest(forw, back, ent.position)
		if samePosition(runTo, back) then
			return false	-- Safer to run backwards
		else
			return true		-- Safer to run forwards
		end
	end
	
	if storage.atomicBombOrArtyResearched == false then
		-- The players don't have atomic bombs yet, so no need to watch out for them.  We can advance.
		return true
	end
	
	-- Look for Atomic Rockets or artillery-projectile
	if game.forces[1].get_entity_count("atomic-rocket") > 0 then
		local rocket = game.surfaces[1].find_entities_filtered{
			position = spider.spiderObject.position,
			radius = 30,
			force = "player",
			name = {"atomic-rocket"},
			limit = 1
		}
		if #rocket > 0 then
			local rocketAngle = rocket[1].orientation * 2 * math.pi + (math.pi / 2)		-- Convert to Radians and rotate 90 degrees
			local intersect = findIntersection(forw, back, rocket[1].position, rocketAngle)
			
			-- Calculate the distance squared from the intersection point to the spider
			local intersectToSpiderDistanceSq = (intersect.x - spider.spiderObject.position.x)^2 + (intersect.y - spider.spiderObject.position.y)^2
			local rocketMaxRangeSq = 16^2	-- We see the rocket after it has already been in flight for an unknown number of frames, so we can only estimate its range
			
			local runFrom = intersect			-- If that distance is less than rocketMaxRangeSq, use strategy 1 - run-from-intercept
			if intersectToSpiderDistanceSq > rocketMaxRangeSq then
				runFrom = rocket[1].position	-- If that distance is more than rocketMaxRangeSq, use strategy 2 - run-from-rocket
			end
			
			-- Run away
			local runTo = furthest(forw, back, runFrom)
			
			-- debugNukeDodge(forw, back, rocket[1].position, rocketAngle, intersect, intersectToSpiderDistanceSq, rocketMaxRangeSq, runFrom, runTo)
			
			if samePosition(runTo, back) then
				return false	-- Safer to run backwards
			else
				return true		-- Safer to run forwards
			end
		end
	end
	
	-- Look for Nuclear Explosions
	local explosion = game.surfaces[1].find_entities_filtered{
		position = spider.spiderObject.position,
		radius = 48,
		name = {"nuke-explosion", "big-artillery-explosion"},
		limit = 1
	}
	if explosion[1] then
		local runTo = furthest(forw, back, explosion[1].position)
		if samePosition(runTo, back) then
			return false	-- Safer to go backwards
		else
			return true		-- Safer to go forwards
		end
	end
	
	-- No scary stuff, advance
	return true
end


function furthest(p1, p2, i3)
	-- We must return either p1 or p2, whichever is furthest from p3
	if i3.x then
		p3 = {x = i3.x, y = i3.y}
	else
		p3 = {x = i3[1], y = i3[2]}
	end
	
	local p1p3 = (p1.x - p3.x)^2 + (p1.y - p3.y)^2
	local p2p3 = (p2.x - p3.x)^2 + (p2.y - p3.y)^2
	if p1p3 > p2p3 then
		return p1
	else
		return p2
	end
end


function findIntersection(forw, back, nuke, angle)
	-- We must return a position containing the intersection point of a line through p1 and p2, with another line through p3 with the given angle
    -- Calculate slopes of the lines
    
	local mNuke = math.tan(angle)		-- Slope of atomic rocket trajectory line
	
	-- Handle edge case where the line defined by forw and back is vertical
    if forw.x == back.x then
        local ix = forw.x
        local mNuke = math.tan(angle)
        local iy = mNuke * (ix - nuke.x) + nuke.y
		-- debugFindIntersection(forw, back, nuke, ix, iy)
        return {x = ix, y = iy}
    end
	
	-- Continue with general case where line is not vertical	
	local mPath = (back.y - forw.y) / (back.x - forw.x)	-- Slope of line connecting forw and back
    
	-- Edge case, if the slopes are equal, these lines are parallel and will never intersect
	if mPath - mNuke == 0 then return furthest(forw, back, nuke) end

    -- Calculate intersection point
    local ix = ((mPath * forw.x - mNuke * nuke.x) + (nuke.y - forw.y)) / (mPath - mNuke)
    local iy = mPath * (ix - forw.x) + forw.y
	-- debugFindIntersection(forw, back, nuke, ix, iy)
    return {x = ix, y = iy}
end


function unstick(spider)
	-- Am I stuck here?
	
	if spider.spiderObject.speed == 0 then
		-- Stuck
		spider.stuckTime = spider.stuckTime + 1
	else
		-- Not stuck
		spider.stuckTime = 0
		return
	end
	
	-- How long have I been stuck?
	if spider.stuckTime < 360 then return end	-- Not that long.
	
	-- Do I have a destination?   I hope so, otherwise I am probably going to stand here forever.
	if spider.spiderObject.autopilot_destinations[1] == nil then return end
	
	-- I've been stuck for a while.
	if spider.pathRequestID then	-- Have I requested a path?
		-- Already requested a path, is it ready?
		if storage.pathRequests[spider.pathRequestID] then	-- My path is ready
			if storage.pathRequests[spider.pathRequestID] == "failed" then
				-- My path request has failed
				-- game.print("Path request failed")
				-- Do something here if we ever find a stuck spider again
			else
				-- My path request succeeded.
				-- Add all the waypoints in the path
				--game.print("storage.pathRequests contained: " .. serpent.line(storage.pathRequests[spider.pathRequestID]))
				
				local oldDestination = spider.spiderObject.autopilot_destinations[1]
				spider.spiderObject.autopilot_destination = nil
				for id, waypoint in pairs(storage.pathRequests[spider.pathRequestID]) do
					spider.spiderObject.add_autopilot_destination(waypoint.position)
				end
				spider.spiderObject.add_autopilot_destination(oldDestination)
				
				storage.pathRequests[spider.pathRequestID] = nil
				spider.stuckTime = 0
			end
		end
		-- If my path isn't ready then just wait I guess

	else
		-- Haven't requested a path yet, so lets do that
		spider.pathRequestID = game.surfaces[1].request_path{
			bounding_box = {{-0.01, -0.01}, {0.01, 0.01}},
			collision_mask = {layers = {object = true, water_tile = true}},
			start = spider.spiderObject.get_spider_legs()[1].position,
			goal = spider.spiderObject.autopilot_destinations[1],
			force = "enemy",
			max_gap_size = 8,
			pathfind_flags = {
				allow_paths_through_own_entities = true,
				no_break = true
			}
		}
		
		--[[
		game.print(
			"Requesting a path from [gps=" .. spider.spiderObject.get_spider_legs()[1].position.x .. "," .. spider.spiderObject.get_spider_legs()[1].position.y ..
			"] to [gps="                   .. spider.spiderObject.autopilot_destinations[1].x     .. "," .. spider.spiderObject.autopilot_destinations[1].y     ..
			"]"
		)
		]]--
	end
end


local on_script_path_request_finished = function(event)
	-- game.print("A path request finished, the event contains: " .. serpent.line(event))
	if event.path then
		storage.pathRequests[event.id] = event.path	-- Path requested succeeded
	else
		storage.pathRequests[event.id] = "failed"	-- Path request failed
	end
end


function interpolate(waypoints)						-- Expects a list of waypoints.  Returns an interpolated list of waypoints
		
	local retVal = {
		{ position = {x = math.floor(waypoints[1][1]), y = math.floor(waypoints[1][2])} }
	}
	
	for wp = 1, #waypoints - 1 do
		local x1 = waypoints[wp][1]
		local y1 = waypoints[wp][2]
		local x2 = waypoints[wp+1][1]
		local y2 = waypoints[wp+1][2]
		
		local numPoints = math.ceil(util.distance(waypoints[wp],waypoints[wp+1]) / storage.interpolationDistance)
		local dx = x2 - x1				-- Difference between X and Y
		local dy = y2 - y1
		local incX = dx / numPoints		-- Amount to move per dimension on each subsequent interpolated point
		local incY = dy / numPoints
		
		for p = 1, numPoints do
			retVal[#retVal+1] = {
				position = {
					x = math.floor(x1 + (incX * p)),
					y = math.floor(y1 + (incY * p))
				}
			}
		end
	end
	
	-- retVal.position = nil
	-- retVal.traversible = nil
	
	return retVal
end


function interpolateWrapped(waypoints)				-- Expects a list of waypoints.  Returns an interpolated list of waypoints that also interpolates from the last waypoint to the first waypoint.
	local waypointsWrapped = waypoints				-- First copy the list of waypoints
	waypointsWrapped[#waypoints+1] = waypoints[1]	-- Then add on a copy the first waypoint to the end
	
	local retVal = {
		{	-- Waypoint 1
			position = {
				x = math.floor(waypoints[1][1]),
				y = math.floor(waypoints[1][2])
			}
		}
	}
	
	for wp = 1, #waypointsWrapped - 1 do
		local x1 = waypointsWrapped[wp][1]
		local y1 = waypointsWrapped[wp][2]
		local x2 = waypointsWrapped[wp+1][1]
		local y2 = waypointsWrapped[wp+1][2]
		
		local numPoints = math.ceil(util.distance(waypointsWrapped[wp],waypointsWrapped[wp+1]) / storage.interpolationDistance)
		local dx = x2 - x1				-- Difference between X and Y
		local dy = y2 - y1
		local incX = dx / numPoints		-- Amount to move per dimension on each subsequent interpolated point
		local incY = dy / numPoints
		
		for p = 1, numPoints do
			retVal[#retVal+1] = {
				position = {
					x = math.floor(x1 + (incX * p)),
					y = math.floor(y1 + (incY * p))
				}
			}
		end
	end
	
	retVal[#retVal] = nil	-- Delete the very last value since it's the same as the first, and we only need the interpolated points up until it wraps around
	return retVal
end


function underlyingObject(sEntity)
	-- Expects a simple entity with force, returns the underlying object
	local found = game.surfaces[1].find_entities_filtered{
		area = {
			left_top = {
				x = sEntity.position.x - 0.5,
				y = sEntity.position.y - 0.5
			},
			right_bottom = {
				x = sEntity.position.x + 0.5,
				y = sEntity.position.y + 0.5
			}
		},
		limit = 1,
		
		invert = true,
		force = {"enemy", "neutral", "fake-player"},
		name = "entity-ghost"
	}
	return found[1]	
end

script.on_event(defines.events.on_entity_damaged,
	function(event)
		-- Transfer damage on the simple-entity-with-force to the underlying object
		if event.entity.force.name == "fake-player" then
			-- game.print(event.tick .. ":   Entity damaged: " .. event.entity.name .. "   Cause: " .. event.cause.name .. "   Source: " .. event.source.name)
			local playerObject = underlyingObject(event.entity)
			if playerObject and playerObject.is_entity_with_health then
				-- game.print("Underlying object name: " .. playerObject.name)
				-- game.print("damage_type: " .. event.damage_type.name .. "   original_damage_amount: " .. event.original_damage_amount)
				local resultingDamage = playerObject.damage(event.original_damage_amount, "enemy", event.damage_type)
				-- game.print("Inflicted " .. resultingDamage .. "   to " .. playerObject.name)
			end
		end
		
		
		-- Mitigation or amplification of damage
		if event.entity.force.name ~= "enemy" then return end	-- This must be a player spider
		if (storage.enemySpiders[event.entity.unit_number] == nil) then return end	-- No data for this spider.  Maybe someone added a spider without giving it properties.
		if event.final_damage_amount == 0 then return end	-- This damage event resulted in 0 damage, no need to perform mitigation on it or record it in the tracking table
		
		--[[ Mitigation Examples:
		1    : 100% damage mitigation
		0.5  :  50% damage mitigation
		0    :   0% dammage mitigation
		
		-0.1 : Amplifies damage by  10%
		-1   : Amplifies damage by 100%
		]]
		
		local spider = storage.enemySpiders[event.entity.unit_number]
		
		if spider.damageTracking.tick ~= event.tick then
			-- Update the tick counter in this spider's damage tracking table
			spider.damageTracking = {tick = event.tick, damageTypes = {}}
			--game.print("Updated the tick counter", {skip = defines.print_skip.never})
		end
		if not spider.damageTracking.damageTypes[event.damage_type.name] then
			-- Add an empty table for this damage type to the damage tracking table, if there's not already a table there
			spider.damageTracking.damageTypes[event.damage_type.name] = {}
			--game.print("Created a new damageType table", {skip = defines.print_skip.never})
		end
		
		--game.print("event.damage_type.name:" .. serpent.line(event.damage_type.name), {skip = defines.print_skip.never})
		--game.print("event.source.unit_number:" .. serpent.line(event.source.unit_number), {skip = defines.print_skip.never})
		--game.print("event.source.name:" .. serpent.line(event.source.name), {skip = defines.print_skip.never})
		--game.print("spider.damageTracking.damageTypes:" .. serpent.line(spider.damageTracking.damageTypes), {skip = defines.print_skip.never})
		--game.print("event.cause.unit_number:" .. serpent.line(event.cause.unit_number), {skip = defines.print_skip.never})
		--game.print("event.cause.name:" .. serpent.line(event.cause.name), {skip = defines.print_skip.never})
		--game.print("event.original_damage_amount:" .. serpent.line(event.original_damage_amount), {skip = defines.print_skip.never})
		--game.print("event.final_damage_amount:" .. serpent.line(event.final_damage_amount), {skip = defines.print_skip.never})
		
		
		local immune = false
		
		if event.cause then
			if spider.damageTracking.damageTypes[event.damage_type.name][event.cause.unit_number] then
				-- If this unit_number already damaged this entity with this damage type on this frame, ignore the damage.
				immune = true
				--game.print("Spider is immune to this damage: ", {skip = defines.print_skip.never})
			end
			if event.source.name == "artillery-projectile"
			or event.source.name == "cannon-projectile"
			or event.source.name == "explosive-cannon-projectile"
			or event.source.name == "uranium-cannon-projectile"
			or event.source.name == "explosive-uranium-cannon-projectile"
			then
				if event.damage_type.name == "explosion" then
					-- Spider should be immune to explosive damages from special weapon types
					immune = true
					--game.print("Special source detected, Immune to this damage: ", {skip = defines.print_skip.never})
				end
			end
			-- Record the source's unit_number or name in the damageSources table			
			spider.damageTracking.damageTypes[event.damage_type.name][event.cause.unit_number] = true
		else
			--game.print("Received damage but there was no cause.", {skip = defines.print_skip.never})
		end
		
	
		--game.print("damageTracking: " .. serpent.line(spider.damageTracking) .. "\n", {skip = defines.print_skip.never})
		
		if event.cause and event.cause.name == "land-mine" and event.cause.force.name == "player" then
			spider.canSeeMines = true
			--game.print("canSeeMines set to True")
		end
		
		local entity_health = event.entity.health	-- This can be 0 if the native damage was enough to "kill" the unit
		if entity_health == 0 then
			entity_health = storage.enemySpiders[event.entity.unit_number].health
		end
		
		local damage_after_native_resists = event.final_damage_amount
		local damage_after_flat_resist = damage_after_native_resists
		
		
		if event.damage_type.name == "fire" and event.force.name == "enemy" then
			damage_after_flat_resist = 0	-- Make spiders immune to fire damage from their own force
			--game.print("I'm immune to fire, muahaha!")
		end
		
		local original_health = entity_health + damage_after_native_resists
		if event.entity.health == 0 then
			original_health = entity_health
		end
		
		local heal_amount = damage_after_flat_resist * storage.enemySpiders[event.entity.unit_number].mitigation
		local damage_to_be_done = math.min(damage_after_flat_resist - heal_amount, storage.damageCap)
		
		if immune then damage_to_be_done = 0 end
		
		local final_health = math.max(original_health - damage_to_be_done, 0)
		
		--[[game.print(
		"   event.entity.health: "         .. event.entity.health ..
		"   event.damage_type.name: "      .. event.damage_type.name ..
		"   event.cause: "      		   .. serpent.line(event.cause) ..
		"   event.source: "      		   .. serpent.line(event.source) ..
		"   event.force.name: "     	   .. serpent.line(event.force.name) ..
		"   damage_after_native_resists: " .. damage_after_native_resists ..
		"   damage_after_flat_resist: "    .. damage_after_flat_resist ..
		"   heal_amount: "                 .. heal_amount                 .. 
		"   damage_to_be_done: "           .. damage_to_be_done           ..
		"   original_health: "             .. original_health             ..
		"   Final_health: "                .. final_health
		)]]
		
		storage.enemySpiders[event.entity.unit_number].health = final_health
		
		event.entity.health = final_health
		
	end
)
script.set_event_filter( defines.events.on_entity_damaged, {
	{filter = "type", type = "spider-vehicle"},
	{mode = "or", filter = "type", type = "simple-entity-with-force"}
} )



local spider = {}
spider.events = {
	[defines.events.on_spider_command_completed]      = on_spider_command_completed,
	[defines.events.on_script_path_request_finished]  = on_script_path_request_finished,
	[defines.events.on_chunk_charted]				  = on_chunk_charted
}
return spider